/*global define */
define(["src/math/Vec2", "src/math/Mat3", "lodash", "src/utils", "src/math/mathUtils", "src/build/VisualMarks",
		"lib/dev", "lib/tasks"],
function (vec2, mat3, lodash, utils, mathUtils, VisualMarks,
		dev, tasks) {
	"use strict";

	var MouseUp = {
		kReturn : 0,
		kHold : 1
	};

	function HandleSelector (aHandle) {
		this.aHandle = aHandle;
		this.tData_Handle = null;
	}

	utils.mixin(HandleSelector, {
		isEmpty : function () {
			return this.aHandle.length < 1;
		},

		setHandleData : function (inData, handle) {
			var id = handle.getStageId();
			this.tData_Handle = this.tData_Handle || {};
			this.tData_Handle[id] = lodash.assign(this.tData_Handle[id], inData);
		},

		getHandleData : function (handle) {
			var id = handle.getStageId();
			if (this.tData_Handle) {
				return this.tData_Handle[id];
			}
		},

		setHandleReleaseData : function (inReleaseTime, inReleaseDuration, handle) {
			this.setHandleData({releaseTime : inReleaseTime, releaseDuration : inReleaseDuration}, handle);
		},

		clearHandleReleaseData : function (handle) {
			this.setHandleReleaseData(null, null, handle);
		},

		getHandleReleaseData : function (handle) {
			var handleData0 = this.getHandleData(handle);
			if (handleData0 && handleData0.releaseDuration !== undefined && handleData0.releaseTime !== undefined) {
				return { releaseDuration: handleData0.releaseDuration, releaseTime: handleData0.releaseTime };
			}
		},

		getHandlePreviouslyHeld : function (handle) {
			var handleData0 = this.getHandleData(handle);
			return (handleData0 && handleData0.previouslyHeldB);
		},

		setHandlePreviouslyHeld : function (handle, inHeldB) {
			this.setHandleData({previouslyHeldB : inHeldB}, handle);
		},

		transformHandleMatrixRelativeToScene : function (inParamBlock, inMatrix, handle) {
			var layerSel = handle.getWarperLayer(),
				matScene_Warper = inParamBlock.getLayerMatrixRelativeToScene(layerSel);
			return mat3.multiply(matScene_Warper, inMatrix);
		},

		setHandleTransform : function (inTransform, handle) {
			this.setHandleData({transform : inTransform}, handle);
		},

		getHandleTransform : function (handle) {
			var	handleData0 = this.getHandleData(handle);
			return handleData0 && handleData0.transform;
		},

		initHandleMatrix : function (inParamBlock, handle, releaseHandleB) {
			var transformData0 = null;
			
			if (!releaseHandleB) {
			//	transformData0 = { matrix: this.getCurrentHandleMatrix(inParamBlock, handle)};
			}
			this.setHandleData({transform : transformData0}, handle);
		},

		resetHandleData : function () {
			delete this.tData_Handle;
		},

		getCurrentHandleMatrix : function (inParamBlock, handle) {
			return inParamBlock.getHandleMatrix(handle);
		},

		getHandleMatrixRelativeToScene : function (inParamBlock, handle) {
			var	mtx = this.getCurrentHandleMatrix(inParamBlock, handle);
			return this.transformHandleMatrixRelativeToScene(inParamBlock, mtx, handle);
		},
		
		// find handle closest to the location
		find : function (inParamBlock, location) {
			var result = this.aHandle.reduce(function (candidate, handle) {

					var warper = handle.getWarperLayer();
					// skip over targets on hidden layers
					if ( !warper.getVisible() ) return candidate;

					var matScene_Handle = this.getHandleMatrixRelativeToScene(inParamBlock, handle),
						position = mat3.getTranslation(matScene_Handle, vec2()),
						offset = position.subtract(location),
						proximity = offset.magnitudeSquared();
						
					if ( !candidate || proximity < candidate.proximity) {
						candidate = candidate || {};
						candidate.proximity = proximity;
						candidate.handle = handle;
						candidate.offset = offset;
					}
					return candidate;
				}.bind(this), null);

			return result;
		}
	});

	function HandleDragger (handleSelector, moverId, handleMover, paramId, paramBlock) {
		this.aHandleMovers = [];
		this.insertHandleMover(handleSelector, moverId, handleMover);
		this.paramId = paramId;
		this.paramBlock = paramBlock;
	}

	utils.mixin(HandleDragger, {
		getHandleMover : function (moverId, inOnlyIfNotReleasedB0) {
			var	hm = this.aHandleMovers[moverId];
			if (inOnlyIfNotReleasedB0 && hm && hm.releasedB) {
				return undefined;
			}
			return hm;
		},
		insertHandleMover : function (handleSelector, moverId, handleMover) {
			var handleMoverHandle0 = handleMover && handleMover.handle, primaryMoverId0, prevAutomation0;
			if (handleMoverHandle0) {
				// Attempt to find this handle already in our list.
				for (var hKey in this.aHandleMovers) {
					if (this.aHandleMovers.hasOwnProperty(hKey)) {
						var handleMover0 = this.aHandleMovers[hKey], h; 
						h = handleMover0 && handleMover0.handle;
						if (h === handleMoverHandle0 && handleMover0) {
							if (!handleMover0.releasedB) {
								primaryMoverId0 = hKey;
							} else if (moverId !== hKey) {
								// delete this released handle since we have a replacement.
								// extract the automation setting in order to correctly
								// persist scale/rotation changes.  -jacquave
								prevAutomation0 = this.aHandleMovers[hKey].automation; 
								delete this.aHandleMovers[hKey];
							}
						}
					}
				}
			}
			
			if (primaryMoverId0) {
				var primaryMover0 = this.getHandleMover(primaryMoverId0);
				if (primaryMover0 && !primaryMover0.secondaryMoverId0) {
					handleMover.primaryMoverId0 = primaryMoverId0;
					primaryMover0.secondaryMoverId0 = moverId;
				} else {
					return;  // Don't add this, since we already have a secondary.
				}
			}
			if (handleMover) {
				if (prevAutomation0) {
					handleMover.automation = prevAutomation0;
				}
				handleMover.releasedB = false; 
			}
			this.aHandleMovers[moverId] = handleMover;
		},
		getTransformHandleData : function (handleSelector, handleMoverHandle, delta, handleLinear) {
			if (handleMoverHandle && delta) {
				var matLayer_Mover = handleSelector.getCurrentHandleMatrix(this.paramBlock, handleMoverHandle),
					matLayer_Target = mat3.multiply(mat3.translation(delta), matLayer_Mover),
					targetLayer = matLayer_Target.getTranslation(),
					targetAnchor = tasks.handle.convertLayerPoint(handleMoverHandle, targetLayer);

				var tdofLayer = { x: targetLayer[0], y: targetLayer[1] }, 
					tdofAnchor = { x: targetAnchor[0], y: targetAnchor[1] };

				if (handleLinear) {
					// also set rotation and scale
					var scale = handleLinear.scale;
					var linear = {
						xScale : scale[0],
						yScale : scale[1],
						angle : handleLinear.rotation
					};

					lodash.assign(tdofLayer, linear);
					lodash.assign(tdofAnchor, linear);
				}

				return {
					matrix : matLayer_Target,
					tdofLayer : tdofLayer,
					tdofAnchor : tdofAnchor
				};
			}
			
			return false;
		},
		getDeltaFromOriginal : function (handleSelector, handleMover, positionVec) {
			var matScene_HandleMover = handleSelector.getHandleMatrixRelativeToScene(this.paramBlock, handleMover.handle),
				ptScene_HandleMover = mat3.getTranslation(matScene_HandleMover),
				delta = positionVec.add(handleMover.offset).subtract(ptScene_HandleMover);
			return delta;
		},

		getLayerDeltaFromOriginal : function (selector, mover, scenePosition) {
			var handle = mover.handle,
				layer = handle.getWarperLayer(),
				sceneTarget = vec2.add(scenePosition, mover.offset),
				matScene_Layer = this.paramBlock.getLayerMatrixRelativeToScene(layer);

			// Remaining coordinates are all relative to Layer (not Scene)
			var target = vec2.transformAffine(mat3.invert(matScene_Layer), sceneTarget),
				origin = selector.getCurrentHandleMatrix(this.paramBlock, mover.handle).getTranslation(),
				delta = target.subtract(origin);

			return delta;
		},

		moveHandle : function (handleSelector, moverId, inCurrentTime, inHeldB, inReleaseDuration, positionVec, getLayerMatrixRelativeToSceneFunc) {
			var handleMover = this.getHandleMover(moverId);
			var handleMoverHandle = handleMover && handleMover.handle;
			if (handleMoverHandle && positionVec && !handleMover.primaryMoverId0) {
				var transformData0, delta;

				delta = this.getLayerDeltaFromOriginal(handleSelector, handleMover, positionVec);						

				if (handleMover.previousScaleVec === undefined) {
					handleMover.previousScaleVec = vec2.initWithArray([1, 1]);
				}
				if (handleMover.scaleVec === undefined) {
					handleMover.scaleVec = vec2.initWithArray([1, 1]);
				}
				if (handleMover.previousRotation === undefined) {
					handleMover.previousRotation = 0;
				}
				if (handleMover.rotation === undefined) {
					handleMover.rotation = 0;
				}
				if (handleMover.automation === undefined) {
					handleMover.automation = { translation : false };
				}

				// This is the primary mover.  Use the secondary to influence scale and rotation.
				if (handleMover.secondaryMoverId0) {
					var secondaryHandleMover0 = this.getHandleMover(handleMover.secondaryMoverId0);

					if (secondaryHandleMover0 && secondaryHandleMover0.handle) {
						if (!secondaryHandleMover0.initialPositionVec) {
							secondaryHandleMover0.initialPositionVec = secondaryHandleMover0.positionVec;
							secondaryHandleMover0.initialPrimaryPositionVec = positionVec;
						}

						var primaryOffset = secondaryHandleMover0.initialPrimaryPositionVec, 
							secondaryOffset = secondaryHandleMover0.initialPositionVec,
							originalDelta, dScale = 0, newDelta = vec2.subtract(secondaryHandleMover0.positionVec, positionVec);

						originalDelta = vec2.subtract(secondaryOffset, primaryOffset);

						dScale = vec2.magnitude(originalDelta);
						if (dScale) {
							handleMover.scaleVec = vec2.scale(vec2.magnitude(newDelta) / dScale, [1, 1]);
						}
						handleMover.rotation = Math.atan2(originalDelta[0], originalDelta[1]) - Math.atan2(newDelta[0], newDelta[1]);
						handleMover.automation.linear = false;
					}
				}

				var handleScaleVec = vec2.xmy(handleMover.previousScaleVec, handleMover.scaleVec),
					handleRotation = handleMover.previousRotation + handleMover.rotation,
					prefixKey = this.paramBlock.getParamEventOutputKey(this.paramId),
					handleKey = prefixKey + this.paramBlock.getParamEventOutputForHandleKey(this.paramId, handleMover.handle);

				var handleLinear = false;
				if (handleMover.automation.linear === false)	{
					handleLinear = {
						scale : handleScaleVec,
						rotation : handleRotation
					};
				}

				transformData0 = this.getTransformHandleData(handleSelector, handleMoverHandle, 
									delta, handleLinear, getLayerMatrixRelativeToSceneFunc);

				this.publishHandleData(handleSelector, handleMoverHandle, handleKey, transformData0, inCurrentTime, inHeldB);

				return true;
			}
			
			return false;
		},
		publishHandleData : function (handleSelector, handle, handleKey, transformData0, inCurrentTime, inHeldB) {
			if (transformData0 && transformData0.tdofLayer) {
				var nowActive = inHeldB ? 1 : 0,
					sustainGraphsB = handleSelector.getHandlePreviouslyHeld(handle) != inHeldB;

				var releaseData0 = handleSelector.getHandleReleaseData(handle);
				if (!inHeldB && releaseData0 && releaseData0.releaseTime !== undefined) {
					if (releaseData0.releaseDuration) {
						nowActive = 1.0 - ((inCurrentTime - releaseData0.releaseTime) / releaseData0.releaseDuration);
						if (nowActive < 0) {
							nowActive = 0;
						} else {
							nowActive = mathUtils.easeInAndOut(nowActive);
						}
					} else {
						nowActive = 1.0;
					}
				}

				this.paramBlock.eventGraph.publish1D(handleKey + "Active", inCurrentTime, nowActive, sustainGraphsB);

				// transformData0 is really an object with matrix *and* tDof components
				handleSelector.setHandleTransform(transformData0, handle);

				// Publish the x and y components
				var tdofLayer = transformData0.tdofLayer,
					tdofAnchor = transformData0.tdofAnchor;

				this.paramBlock.eventGraph.publish1D(handleKey + "Translation/Layer/X", inCurrentTime, tdofLayer.x, sustainGraphsB);
				this.paramBlock.eventGraph.publish1D(handleKey + "Translation/Layer/Y", inCurrentTime, tdofLayer.y, sustainGraphsB);
				this.paramBlock.eventGraph.publish1D(handleKey + "Translation/Anchor/X", inCurrentTime, tdofAnchor.x, sustainGraphsB);
				this.paramBlock.eventGraph.publish1D(handleKey + "Translation/Anchor/Y", inCurrentTime, tdofAnchor.y, sustainGraphsB);
				
				if (tdofLayer.angle) {
					// either layer or anchor will do -- they are same
					var tdof = tdofLayer;
					this.paramBlock.eventGraph.publish1D(handleKey + "Scale/X", inCurrentTime, tdof.xScale, sustainGraphsB);
					this.paramBlock.eventGraph.publish1D(handleKey + "Scale/Y", inCurrentTime, tdof.yScale, sustainGraphsB);
					this.paramBlock.eventGraph.publish1D(handleKey + "Angle/Radians", inCurrentTime, tdof.angle, sustainGraphsB);
					this.paramBlock.eventGraph.publish1D(handleKey + "Angle/IsSet", inCurrentTime, 1, true);
				} else {
					this.paramBlock.eventGraph.publish1D(handleKey + "Angle/IsSet", inCurrentTime, 0, true);
				}
				handleSelector.setHandlePreviouslyHeld(handle, inHeldB);
			} else {
				this.paramBlock.eventGraph.publish1D(handleKey + "Active", inCurrentTime, 0.0, true);
			}

		},
		setHandlePos : function (moverId, positionVec) {
			var handleMover = this.getHandleMover(moverId);
			var handleMoverHandle = handleMover && handleMover.handle;
			if (handleMoverHandle && positionVec) {
				handleMover.positionVec = positionVec;
			}
		},
		publishHandles : function (handleSelector, inCurrentTime, inOnMouseUp, inReleaseDuration, getLayerMatrixRelativeToSceneFunc) {
			var	hKey, handleMover, aHandlesMoved = [], that = this;
			
			for (hKey in this.aHandleMovers) {
				if (this.aHandleMovers.hasOwnProperty(hKey)) {
					handleMover = this.aHandleMovers[hKey];
					if (handleMover && handleMover.positionVec && !handleMover.releasedB) {
						if (this.moveHandle(handleSelector, hKey, inCurrentTime, true, inReleaseDuration,
											handleMover.positionVec, getLayerMatrixRelativeToSceneFunc)) {
							aHandlesMoved[handleMover.handle.getDagPathId()] = true;
						}
					}
				}
			}
			
			handleSelector.aHandle.forEach(function (handle) {
				if (!aHandlesMoved[handle.getDagPathId()]) {
					// Handle not moved, so publish it's state.
					var prefixKey = that.paramBlock.getParamEventOutputKey(that.paramId),
						handleKey = prefixKey + that.paramBlock.getParamEventOutputForHandleKey(that.paramId, handle),
						handleTransformData0 = handleSelector.getHandleTransform(handle);
					
					that.publishHandleData(handleSelector, handle, handleKey, handleTransformData0, inCurrentTime, false);
				}
			});
		},
		releaseHandle : function (moverId, handleSelector, inParamBlock, onMouseUp, inReleaseDuration) {
			var handleMover = this.getHandleMover(moverId),
				handleMoverHandle = handleMover && handleMover.handle;

			if (handleMover && !handleMover.releasedB) {
				if (handleMoverHandle && !handleMover.primaryMoverId0) {
					if (onMouseUp === MouseUp.kHold) {
						inReleaseDuration = 0.0;
					} else {
						// Set this to a very small number in order to have it release and not hold.  -jacquave
						inReleaseDuration = Math.max(Number.MIN_VALUE, inReleaseDuration);
					}
					handleSelector.setHandleReleaseData(inParamBlock.currentTime, inReleaseDuration, handleMover.handle);
					this.aHandleMovers[moverId].releasedB = true;
				}

				if (handleMover && handleMover.primaryMoverId0) {
					// This is a secondary mover.
					var primaryMover0 = this.getHandleMover(handleMover.primaryMoverId0);
					if (primaryMover0 && !primaryMover0.releasedB) {
						if (primaryMover0.previousRotation !== undefined && 
								primaryMover0.rotation !== undefined) {
							primaryMover0.previousRotation += primaryMover0.rotation;
							delete primaryMover0.rotation;
						}
						if (primaryMover0.previousScaleVec !== undefined && 
								primaryMover0.scaleVec !== undefined) {
							primaryMover0.previousScaleVec = vec2.xmy(primaryMover0.previousScaleVec, primaryMover0.scaleVec);
							delete primaryMover0.scaleVec;
						}
						delete primaryMover0.secondaryMoverId0;
					}
					delete this.aHandleMovers[moverId];
				}
			}
		},
		releaseAllHandles : function (handleSelector, inParamBlock, onMouseUp, inReleaseDuration, exclusionKeysArray0) {
			var	hKey;
			
			for (hKey in this.aHandleMovers) {
				if (this.aHandleMovers.hasOwnProperty(hKey)) {
					if (!exclusionKeysArray0 || !exclusionKeysArray0[hKey]) {
						this.releaseHandle(hKey, handleSelector, inParamBlock, onMouseUp, inReleaseDuration);
					}
				}
			}
		},
		getXYParamEventValue : function (inParamBlock, inEvaluator, inInputParamId, inEventGraphKey, inHandleKey) {
			var xy = [];
			xy[0] = inParamBlock.getParamEventValueFromEvaluator(inEvaluator, inInputParamId, inEventGraphKey + "/X");
			xy[1] = inParamBlock.getParamEventValueFromEvaluator(inEvaluator, inInputParamId, inEventGraphKey + "/Y");
			if (xy[0] === undefined || xy[1] === undefined) {
				xy = undefined;
			}
			return xy;
		},
		updateHandles : function (inHandleSelector, inParamBlock, inInputParamId) {
			var that = this;
			inHandleSelector.aHandle.forEach(function (handle) {
				
				var activeValue,
					prefixKey = inParamBlock.getParamEventOutputKey(inInputParamId),
					handleKey = inParamBlock.getParamEventOutputForHandleKey(inInputParamId, handle),
					eventGraphKey = prefixKey + handleKey,
					graphEvaluatorArray = inParamBlock.getGraphEvaluatorArray(inInputParamId, handleKey),
					graphEvaluatorWeights = inParamBlock.getEvaluatorsValueWeightData(inInputParamId, true, graphEvaluatorArray);
				
				for (var idx = 0; idx < (graphEvaluatorArray && graphEvaluatorArray.length); idx += 1) {
					var evaluator = graphEvaluatorArray[idx];

					activeValue = inParamBlock.getParamEventValueFromEvaluator(evaluator, inInputParamId, eventGraphKey + "Active");

					if (activeValue !== undefined) {

						var posAnchor,
							tdof = {},
							relativeToAnchor = inParamBlock.getParam("relativeToAnchor"),
							evalWeight = graphEvaluatorWeights.weightValueArray[idx];
						
						activeValue *= evalWeight.weight;
						if (relativeToAnchor) {
							// use Anchor coordinates...
							posAnchor = that.getXYParamEventValue(inParamBlock, evaluator, inInputParamId, eventGraphKey + "Translation/Anchor");
						} else {
							// use Layer coordinates...
							var posLayer = that.getXYParamEventValue(inParamBlock, evaluator, inInputParamId, eventGraphKey + "Translation/Layer");
							if ( !lodash.isUndefined(posLayer) ) posAnchor = tasks.handle.convertLayerPoint(handle, posLayer);
						}

						if (!lodash.isUndefined(posAnchor)) lodash.assign(tdof, { x: posAnchor[0], y: posAnchor[1] });

						var angleIsSet = inParamBlock.getParamEventValueFromEvaluator(evaluator, inInputParamId, eventGraphKey + "Angle/IsSet"),
							angle = inParamBlock.getParamEventValueFromEvaluator(evaluator, inInputParamId, eventGraphKey + "Angle/Radians"),
							xScale = inParamBlock.getParamEventValueFromEvaluator(evaluator, inInputParamId, eventGraphKey + "Scale/X"),
							yScale = inParamBlock.getParamEventValueFromEvaluator(evaluator, inInputParamId, eventGraphKey + "Scale/Y");

						if ( !lodash.isUndefined(angleIsSet) && angleIsSet === 1) {
							lodash.assign(tdof, {
								xScale : xScale,
								yScale : yScale,
								angle : angle
							});
						}

						// issue task when necessary
						if ( !lodash.isEmpty(tdof) && activeValue > 0.0
								// && activeValue 	// Note, I think we should still set a task for zero based weights in order for
													// blending with other values to be correctly averaged and continuous.  */
						   		) {
							var move = new tasks.MoveTo(tdof);
							// console.log("Dragger Task added for " + eventGraphKey + ", pos=" + tdof.x + "," + tdof.y + ", angle=" + tdof.angle + "; weight=" + activeValue);
							tasks.handle.attachTask(handle, move, activeValue);
							inParamBlock.setEventGraphParamRecordingValid(inInputParamId, handleKey);
						}			

					}
				}
			});
		}
	});
	
	function showTargets (self, inParamBlock) {
		var markSize = 20,
			markOpacity = 0.5;

		if (inParamBlock.getParam("showTargets")) {
			if (self.targetMarks) {
				// update
				self.targetMarks.forEach(function (mark, index) {
					mark.setMatrix(inParamBlock.getHandleMatrixRelativeToScene(self.targets[index]));
				});
			} else {
				// create
				self.targetMarks = new VisualMarks({ opacity : markOpacity });
				self.targets.forEach(function (hi, index) {
					var mark = self.targetMarks.square(markSize, 1);
					self.targetMarks.insertMark(mark);
					mark.setMatrix(inParamBlock.getHandleMatrixRelativeToScene(self.targets[index]));
				});

				self.targetMarks.addToDisplayItem(inParamBlock.stageLayer.privateLayer.getTrackItem());
			}
		} else {
			// remove
			if (self.targetMarks) {
				self.targetMarks.removeFromDisplayItem(inParamBlock.stageLayer.privateLayer.getTrackItem());
				self.targetMarks = null;
			}
		}
	}


	return {
		about: "$$$/private/animal/Behavior/Dragger/About=Dragger, (c) 2014.",
		description: "$$$/animal/Behavior/Dragger/Desc=Tracks mouse and touch screen motion",
		uiName: 	"$$$/animal/Behavior/Dragger/UIName=Dragger",
		defaultArmedForRecordOn: true,

		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			var params = [
				{ 	id: "Input",
					type: "eventGraph", 
					uiName: "$$$/animal/Behavior/Dragger/Parameter/Input=Mouse & Touch Input",
					inputKeysArray: ["Mouse/", "Touch/"], 
					supportsBlending: true,
					uiToolTip: "$$$/animal/Behavior/Dragger/Parameter/Input/tooltip=Mouse and Touch input used to transform handles", 
					defaultArmedForRecordOn: true,
					outputKeyTraits : {
						// Future definition of takeLeaves.  Requires "tasks" to correctly
						// organize the graphs this way.  -jacquave
						/*
						takeLeaves : {
							Active : {    
								type : "boolean",
								presentation : "temporalBars",
								uiName : "$$$/animal/Behavior/Dragger/takeLeafName/Active=Active"
							},
							Position : {    
								type : "spatial2D",
								presentation : "spatialHandle:{*}",
								uiName : "$$$/animal/Behavior/Dragger/takeLeafName/Position=Position"
							},
							Rotation : {    
								type : "temporal1D",
								presentation : "temporalGraph",
								uiName : "$$$/animal/Behavior/Dragger/takeLeafName/Rotation=Rotation"
							},
							Scale : {    
								type : "temporal1D",
								presentation : "temporalGraph",
								uiName : "$$$/animal/Behavior/Dragger/takeLeafName/Scale=Scale"
							}
						},
						*/
						takeGroupsArray : [
							{    
								id : "HandleTarget/*",
								uiName : "$$$/animal/Behavior/Dragger/takeGroupName=Handle ({handleName:*})",
								/* {} specifies a functional substitution where handleName will be a function invoked with the argument, the * from the take group id. */
								/* Future...
								children : [ 
									{ leafId : "Active"},
									{ leafId : "Position"},
									{ leafId : "Rotation"},
									{ leafId : "Scale"}
								]
								*/
							}
						]
					 }
				},
				{
					id: "targets",
					type: "handle",
					uiName: "$$$/animal/Behavior/Dragger/Parameter/targets=Target Handles",
					dephault: { match: "//Adobe.Dragger.Draggable" }
				},
				{
					id : "showTargets",
					type : "checkbox",
					uiName: "$$$/animal/Behavior/Dragger/Parameter/showTargets=Show Targets",
					hidden : true,
					dephault : false
				},
				{
					id: "onMouseUp",   
					type:"enum",
					uiName: "$$$/animal/Behavior/Dragger/Parameter/onMouseUp=After Move",
					items: [
						{ id: MouseUp.kReturn, uiName: "$$$/animal/Behavior/Dragger/Parameter/onMouseUp/Return=Return to Rest" },
						{ id: MouseUp.kHold, uiName: "$$$/animal/Behavior/Dragger/Parameter/onMouseUp/Hold=Hold in Place" }
					],
					hideRecordButton: true,
					dephault: MouseUp.kReturn
				},
				{ 	id: "handleReleaseDuration", type: "slider", uiName: "$$$/animal/Behavior/Dragger/Parameter/handleReleaseDuration=Return Duration",
				 	uiUnits: "$$$/animal/Behavior/Dragger/units/sec=sec", min: 0, max: 10, precision: 2, dephault: 0.0, hideRecordButton: true,
					uiToolTip: "$$$/animal/behavior/Dragger/param/handleReleaseDuration/tooltip=How long it takes the dragged handle to return to rest" 
				},
				{
					id : "relativeToAnchor",
					type : "checkbox",
					uiName: "$$$/animal/Behavior/Dragger/Parameter/relativeToAnchor=Record in Parent Frame",
					hidden : true,
					dephault : false
				}				
			];

			return params;
		},

		defineTags: function () {
			return { 
				aTags: [
					// {
					// 	id: "Adobe.Tag.Thing",
					//  artMatches: array of synonyms (even x-language)
					//  uiName: zstr with presentation name
					//  uiToolTip: "DaveS loves tooltips",
					//	tagType: "handletag" or "layertag"
					//  params: [ {use same param definition as behavior} ]
					// 
					// },
					{
						id: "Adobe.Dragger.Draggable",
						artMatches: ["draggable", "mousetrack"],
						uiName: "$$$/animal/Behavior/Draggable/UIName/Draggable=Draggable",
						tagType: "handletag",
						uiGroups: [{ id:"Adobe.TagGroup.Modifiers"}]						
					}
				]
			};
		},


		onCreateBackStageBehavior: function (/*self*/) {
		},

		onCreateStageBehavior: function (self, args) {
			self.targets = args.getStaticParam("targets");
			self.handleSelector = new HandleSelector(self.targets);
			self.getLayerMatrixRelativeToScene = args.getLayerMatrixRelativeToScene;
			self.showSelector = {
				display : false,
				marks : false
			};
		},

		dragHandleFromInput: function (self, args, inInputKey, inParamId, inPosVec, outMovedHandleInputKeys) {
			var	foundHandleMover0;
			if (!self.dragger && inPosVec) {
				foundHandleMover0 = self.handleSelector.find(args, inPosVec);
				self.dragger = new HandleDragger(self.handleSelector, inInputKey, foundHandleMover0, inParamId, args);
			} else {
				if (self.dragger && !self.dragger.getHandleMover(inInputKey, true)) {
					foundHandleMover0 = self.handleSelector.find(args, inPosVec);
					self.dragger.insertHandleMover(self.handleSelector, inInputKey, foundHandleMover0);
				}
			}
			if (foundHandleMover0 && foundHandleMover0.handle && !foundHandleMover0.primaryMoverId0) {
				self.handleSelector.initHandleMatrix(args, foundHandleMover0.handle, false);
				self.handleSelector.clearHandleReleaseData(foundHandleMover0.handle);
			}

			// move selected handle
			if (self.dragger) {
				self.dragger.setHandlePos(inInputKey, inPosVec);
				outMovedHandleInputKeys[inInputKey] = true;
			}
		},

		// Clear the rehearsal state
		onResetRehearsalData : function (self) {
			self.dragger = null;
			self.handleSelector.resetHandleData();
		},

		// Filter live input parameters into Behavior Actions
		onFilterLiveInputs: function (self, args) {
			if (self.handleSelector.isEmpty()) return;

			var inputParamId = "Input", inputLiveB = args.isParamEventLive(inputParamId),
					movedHandleKeysA = [], onMouseUp = args.getParam("onMouseUp"), 
					releaseDuration = args.getParam("handleReleaseDuration");
			
			if (false && self.onMouseUp !== onMouseUp) {
				this.onResetRehearsalData(self);
				// Unpublish our specific data.  The engine will do this when it
				// decides the rehearsal state has changed, but we invoke this on our own
				// and need to do this directly when onMouseUp changes.
				args.eventGraph.removeGraphs(args.getParamEventOutputKey(inputParamId));
				self.onMouseUp = onMouseUp;
			}
			if (inputLiveB) {
				var leftDownB = args.getParamEventValue(inputParamId, "Mouse/Down/Left", null, 0, true),
					mousePosition0 = args.getParamEventValue(inputParamId, "Mouse/Position", null, [0,0], true);

				// Publish handles using mouse events
				if (leftDownB) {
					var mouseVec;
					if (mousePosition0) mouseVec = vec2(mousePosition0);
					this.dragHandleFromInput(self, args, "Mouse", inputParamId, mouseVec, movedHandleKeysA);
				}

				// Publish handles using touch events
				var touchEvents = args.getParamEventValue(inputParamId, "Touch/IndexToID/Count", null, 0, true),
						touchIdx, touchId, touchIdKey, touchPos, touchDown;

				if (touchEvents) {
					for (touchIdx = 0; touchIdx < touchEvents; touchIdx += 1) {
						touchId = args.getParamEventValue(inputParamId, "Touch/IndexToID/" + touchIdx, null, 0, true);
						if (touchId !== undefined) {
							touchIdKey = "Touch/ID/" + touchId;
							touchDown = args.getParamEventValue(inputParamId, touchIdKey + "/Down", null, 0, true);
							if (touchDown) {
								touchPos = args.getParamEventValue(inputParamId, touchIdKey + "/Position", null, [0,0], true);
								if (touchPos) {
									var touchVec = vec2(touchPos);
									this.dragHandleFromInput(self, args, touchIdKey, inputParamId, touchVec, movedHandleKeysA);
								}
							}
						}
					}
				}
				if (self.dragger) {
					self.dragger.releaseAllHandles(self.handleSelector, args, onMouseUp, releaseDuration, movedHandleKeysA);
					self.dragger.publishHandles(self.handleSelector, args.currentTime, onMouseUp, releaseDuration, args.getLayerMatrixRelativeToScene);
				}
			}
		},

		// Animate the Behavior
		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			if (self.handleSelector.isEmpty()) return;

			showTargets(self, args);

			var inputParamId = "Input";
			
			HandleDragger.prototype.updateHandles(self.handleSelector, args, inputParamId);
		}

	}; // end of object being returned
});
